﻿/*
Copyright 2018 item Industrietechnik GmbH [http://www.item24.com]

This license is for the sample source code only:
----------------------------------------------------------------------------
   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
----------------------------------------------------------------------------

The compiled libraries that come with this sample have their own license
attached to them. Make sure you understand those licenses.

ObjectService can ONLY be legally used together with item ILMU hardware and
ONLY for the creation of software used to control or analyze purchased item
hardware. It may NOT under any circumstances be transmitted to other companies
or entities unless prior permission was granted by item Industrietechnik GmbH.
*/

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Windows;
using System.Windows.Media;

///////////////////////////////////////////////////////////////////////////////////////////////
// WARNING - WARNING - WARNING - WARNING - WARNING - WARNING - WARNING - WARNING - WARNING - 
///////////////////////////////////////////////////////////////////////////////////////////////
// DO NOT use this software on an unparametrized ILMU, because it may be destroyed
// Make sure the ILMU is ready to be moved and that the machine is SAFE
// Use the MotionSoft setup routine if you are not sure!
//
// To write into ObjectService registers, you can use the following code:
// Controller.WriteComObject(Item.Ilmu.ObjectService.ComObjectType.BETRIEBSART, (int)Item.Ilmu.ObjectService.BetriebsArt.Positioning);
//
// DO NOT WRITE INTO ANY CONTROLLER REGISTERS IF YOU ARE NOT 100% SURE WHAT THEY DO!
// Writing incorrect data into the registers will destroy the ILMU hardware if you try to move
///////////////////////////////////////////////////////////////////////////////////////////////
// WARNING - WARNING - WARNING - WARNING - WARNING - WARNING - WARNING - WARNING - WARNING - 
///////////////////////////////////////////////////////////////////////////////////////////////

namespace CSharp_MoveSample
{
    public partial class MainWindow : Window
    {
        #region Variables

        // One ObjectService. You can instantiate more than one if you have multiple axis
        Item.Ilmu.ObjectService.ObjectService Controller;

        uint LastStatusWord = 0;
        #endregion

        #region Constructor / Destructor

        public MainWindow()
        {
            InitializeComponent();

            // Instantiating the ObjectService
            //
            ///////////////////////////////////////////////////////////////////////////////////////////////
            // WARNING - WARNING - WARNING - WARNING - WARNING - WARNING - WARNING - WARNING - WARNING - 
            ///////////////////////////////////////////////////////////////////////////////////////////////
            //
            // ALWAYS instantiate using either:
            //
            // ObjectServiceMode.OnlyMovement_ReadWrite - Use this if you want to move the axis using your software (write access to movement variables)
            // ObjectServiceMode.JustWatching_ReadOnly - Use this if you just need to read certain variables
            //
            // That way, you can only change the variables that have to do with movement or reading other variables.
            //
            // If you do instantiate with the other option, you will have full control over ALL variables, and there
            // is a pretty high chance you will destroy your hardware when developing your software.
            // Using the third option IS NOT CLEARED FOR USAGE BY CUSTOMERS, because a wrong order of commands or
            // usage of wrong options can destroy the hardware or hurt people.
            //
            // NOTE: Even in the movement mode you CAN destroy your axis if you move it to somewhere unsafe and have not properly secured it with limit switches and a good MotionSoft setup
            // NOTE: Always make sure you do not exceed the weight limits of your hardware. If the maximum current can not hold the weight, the axis may suddenly drop down and do extensive damage or hurt people.
            // NOTE: In case it is not clear by now, using the ObjectService DLL is dangerous in any case, as there is no built in safety that will prevent you from doing very bad things.

            Controller = new Item.Ilmu.ObjectService.ObjectService(Item.Ilmu.ObjectService.ObjectServiceMode.OnlyMovement_ReadWrite, Item.Ilmu.ObjectService.LoggingMode.Debug | Item.Ilmu.ObjectService.LoggingMode.Info | Item.Ilmu.ObjectService.LoggingMode.Success | Item.Ilmu.ObjectService.LoggingMode.Warn, false);

            // Some of the events you can subscribe to
            Controller.Connected += Controller_Connected;
            Controller.Disconnected += Controller_Disconnected;
            Controller.LogMessage += Controller_LogMessage;
            Controller.MovementStatusChanged += Controller_MovementStatusChanged;
            Controller.RegisterChanged += Controller_RegisterChanged;
            Controller.ErrorsReported += Controller_ErrorsReported;
            Controller.ConnectStatusChanged += Controller_ConnectStatusChanged;

            // You need to subscribe to variables if you want to get updates every X milliseconds through the event "RegisterChanged"
            Controller.Subscribe(new List<Item.Ilmu.ObjectService.SubscriptionRequest>()
            {
                new Item.Ilmu.ObjectService.SubscriptionRequest() { ComObject = Item.Ilmu.ObjectService.ComObjectType.SPEEDACTUAL, IntervalInMS = 250, SendHistory = false },
                new Item.Ilmu.ObjectService.SubscriptionRequest() { ComObject = Item.Ilmu.ObjectService.ComObjectType.TEMPMOTOR, IntervalInMS = 250, SendHistory = false },
                new Item.Ilmu.ObjectService.SubscriptionRequest() { ComObject = Item.Ilmu.ObjectService.ComObjectType.CURRENTACTUAL, IntervalInMS = 250, SendHistory = false },
                new Item.Ilmu.ObjectService.SubscriptionRequest() { ComObject = Item.Ilmu.ObjectService.ComObjectType.TEMPLEISTUNG, IntervalInMS = 250, SendHistory = false },
                new Item.Ilmu.ObjectService.SubscriptionRequest() { ComObject = Item.Ilmu.ObjectService.ComObjectType.TORQUEACTUAL, IntervalInMS = 250, SendHistory = false },
                new Item.Ilmu.ObjectService.SubscriptionRequest() { ComObject = Item.Ilmu.ObjectService.ComObjectType.TEMPKONDENSATOR, IntervalInMS = 250, SendHistory = false },
                new Item.Ilmu.ObjectService.SubscriptionRequest() { ComObject = Item.Ilmu.ObjectService.ComObjectType.WIRKLEISTUNGACTUAL, IntervalInMS = 250, SendHistory = false },
                new Item.Ilmu.ObjectService.SubscriptionRequest() { ComObject = Item.Ilmu.ObjectService.ComObjectType.I2TMOTOR, IntervalInMS = 250, SendHistory = false },
                new Item.Ilmu.ObjectService.SubscriptionRequest() { ComObject = Item.Ilmu.ObjectService.ComObjectType.BETRIEBSART, IntervalInMS = 250, SendHistory = false },
                new Item.Ilmu.ObjectService.SubscriptionRequest() { ComObject = Item.Ilmu.ObjectService.ComObjectType.VOLTZWISCHENKREIS, IntervalInMS = 250, SendHistory = false },
            });
        }
        private void Window_Closing(object sender, System.ComponentModel.CancelEventArgs e)
        {
            // If you dont disconnect the ObjectService thread will stay active and you cant close your application
            Controller.Disconnect();
        }

        #endregion

        #region ObjectService Callbacks

        /// <summary>
        /// A list with error messages. There also are other ways to get errors, like:
        /// Controller.CurrentErrorCodes
        /// Controller.HasError
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Controller_ErrorsReported(Item.Ilmu.ObjectService.ObjectService sender, List<string> e)
        {
            Debug.WriteLine("Errors: " + String.Join(", ", e));
        }

        /// <summary>
        /// This event will list ANY register change in the ObjectService. There are lots of them. Look at the enum "Item.Ilmu.ObjectService.ComObjectType" for a complete list.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="register"></param>
        private void Controller_RegisterChanged(Item.Ilmu.ObjectService.ObjectService sender, Item.Ilmu.ObjectService.ComObjectValue register)
        {
            switch (register.ID)
            {
                case Item.Ilmu.ObjectService.ComObjectType.STATUSWORD:

                    uint Status = register.GetValue<uint>();

                    //if (Status != LastStatusWord)
                    //    Debug.WriteLine("ILMU Status: " + ((Item.Ilmu.ObjectService.StatusWord)Status).ToString());

                    LastStatusWord = Status;
                    break;

                case Item.Ilmu.ObjectService.ComObjectType.TEMPMOTOR:

                    //Debug.WriteLine("Temp Motor (°C): " + register.GetValue<double>().ToString("0.0"));
                    break;

                case Item.Ilmu.ObjectService.ComObjectType.CURRENTACTUAL:

                    //Debug.WriteLine("Current (A): " + register.GetValue<double>().ToString("0.0"));
                    break;

                case Item.Ilmu.ObjectService.ComObjectType.SPEEDACTUAL:

                    //Debug.WriteLine("Speed (mm/s): " + register.GetValue<double>().ToString("0.0"));
                    break;

                case Item.Ilmu.ObjectService.ComObjectType.TEMPLEISTUNG:

                    //Debug.WriteLine("Temp Controller (°C): " + register.GetValue<double>().ToString("0.0"));
                    break;

                case Item.Ilmu.ObjectService.ComObjectType.WIRKLEISTUNGACTUAL:

                    //Debug.WriteLine("Effective power (W): " + register.GetValue<double>().ToString("0.0"));
                    break;

                case Item.Ilmu.ObjectService.ComObjectType.I2TMOTOR:

                    //Debug.WriteLine("I2T (%): " + register.GetValue<double>().ToString("0.0"));
                    break;

                case Item.Ilmu.ObjectService.ComObjectType.VOLTZWISCHENKREIS:

                    //Debug.WriteLine("Intermediate circuit voltage (V): " + register.GetValue<double>().ToString("0.0"));
                    break;

                case Item.Ilmu.ObjectService.ComObjectType.POSACTUAL:

                    //Debug.WriteLine("Position: " + register.GetValue<double>().ToString("0.0"));
                    break;
            }
        }

        /// <summary>
        /// This will show changes in the movement process
        /// </summary>
        /// <param name="sender"></param>
        private void Controller_MovementStatusChanged(Item.Ilmu.ObjectService.ObjectService sender)
        {
            Debug.WriteLine("MoveStatus: " + sender.MovementStatus.ToString() + " | MoveMode: " + sender.MovementMode.ToString());
        }

        /// <summary>
        /// Lots of information is logged through this. Set up 'Item.Ilmu.ObjectService.LoggingMode' during construction of the ObjectService instance
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Controller_LogMessage(Item.Ilmu.ObjectService.ObjectService sender, Item.Ilmu.ObjectService.Logger.LogMessageEventArgs e)
        {
            Debug.WriteLine(e.Message + " [" + ((Item.Ilmu.ObjectService.LogErrorCodes)e.ErrorCode).ToString() + "]");
        }

        /// <summary>
        /// This event will tell you if the ObjectService is reconnecting internally
        /// </summary>
        /// <param name="sender"></param>
        private void Controller_ConnectStatusChanged(Item.Ilmu.ObjectService.ObjectService sender)
        {
            Debug.WriteLine("Connect Status: " + sender.ComStatus.ToString());

            UpdateConnectionStatus();
        }

        /// <summary>
        /// This will fire if the connection to the controller is lost
        /// </summary>
        /// <param name="sender"></param>
        private void Controller_Disconnected(Item.Ilmu.ObjectService.ObjectService sender)
        {
            Debug.WriteLine("ILMU Disconnected");

            UpdateConnectionStatus();
        }

        /// <summary>
        /// This will fire if a connection was established
        /// </summary>
        /// <param name="sender"></param>
        private void Controller_Connected(Item.Ilmu.ObjectService.ObjectService sender)
        {
            Debug.WriteLine("ILMU Connected");

            UpdateConnectionStatus();
        }

        #endregion

        #region Interface Callbacks

        private async void ButtonConnect_Click(object sender, RoutedEventArgs e)
        {
            // When set to AUTO, the connection routine tests across USB, Serial Port and UDP by itself
            await Controller.ConnectAsync(Item.Ilmu.ObjectService.ComDeviceType.AUTO, "192.168.0.82", 8802, "AUTO", 0);

            // There is an ASYNC and a BLOCKING call for most functions.
            //
            // Just remove the Async from the function name if you do not want to use the C# Task pattern
            // It is a very bad idea to do this, unless you are using a singlethreaded programming language
            // like Python or other script languages. If you use C#, stick with the Async pattern so you
            // do not prevent your interface thread from running and updating
            //
            //Controller.Connect(Item.Ilmu.ObjectService.ComDeviceType.AUTO, "192.168.0.82", 8802, "AUTO", 0);
        }

        private async void ButtonDisconnect_Click(object sender, RoutedEventArgs e)
        {
            await Controller.DisconnectAsync();

            //Controller.Disconnect();
        }

        private async void ButtonMove_Click(object sender, RoutedEventArgs e)
        {
            double Position = 0;
            double Speed = 50;
            double Acceleration = 3000;
            double TorqueLimit = 0.8;
            bool HasTorqueLimit = (CheckboxTorqueLimited.IsChecked == true);

            try
            {
                Position = double.Parse(TextboxPosition.Text.Replace(',', '.'), System.Globalization.CultureInfo.InvariantCulture);
            }
            catch (Exception)
            {
            }

            try
            {
                Speed = double.Parse(TextboxSpeed.Text.Replace(',', '.'), System.Globalization.CultureInfo.InvariantCulture);
            }
            catch (Exception)
            {
            }

            try
            {
                Acceleration = double.Parse(TextboxAcceleration.Text.Replace(',', '.'), System.Globalization.CultureInfo.InvariantCulture);
            }
            catch (Exception)
            {
            }

            try
            {
                TorqueLimit = double.Parse(TextboxLimit.Text.Replace(',', '.'), System.Globalization.CultureInfo.InvariantCulture);
            }
            catch (Exception)
            {
            }

            // How to read register values from the Controller
            // You need to specify the correct variable type, otherwise it will not work
            ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
            // These variables should be set up in MotionSoft, DO NOT use this software on an unparametrized ILMU, because it may be destroyed
            ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
            double SoftwareEndswitchMaximum = await Controller.GetComObjectValueAsync<double>(Item.Ilmu.ObjectService.ComObjectType.SWENDSWITCHMAXIMUMH);
            double SoftwareEndswitchMinimum = await Controller.GetComObjectValueAsync<double>(Item.Ilmu.ObjectService.ComObjectType.SWENDSWITCHMINIMUMH);
            //double SoftwareEndswitchMaximum = Controller.GetComObjectValue<double>(Item.Ilmu.ObjectService.ComObjectType.SWENDSWITCHMAXIMUMH);
            //double SoftwareEndswitchMinimum = Controller.GetComObjectValue<double>(Item.Ilmu.ObjectService.ComObjectType.SWENDSWITCHMINIMUMH);

            // Clamp to allowed values minus 0.5, in case the machine is configured to to an immediate stop when reaching software limits
            Position = Math.Min(SoftwareEndswitchMaximum - 0.5, Math.Max(SoftwareEndswitchMinimum + 0.5, Position));

            Item.Ilmu.ObjectService.LogErrorCodes f;

            if (HasTorqueLimit)
            {
                f = await MovementMethod1_ManualWithTorqueLimit(Position, Speed, Acceleration, TorqueLimit);
            }
            else
            {
                f = await MovementMethod2_AutomaticPositioning(Position, Speed, Acceleration);
            }

            Debug.WriteLine("Movement Result: " + f.ToString());
        }

        /// <summary>
        /// This will move to the position by storing it in WP 0 and then starting the waypoint program from WP 0. If it hits the torque limit it will immediately go to WP 1, which is at position zero.
        /// </summary>
        /// <param name="Position"></param>
        /// <param name="Speed"></param>
        /// <param name="Acceleration"></param>
        /// <param name="TorqueLimit"></param>
        /// <returns></returns>
        private async System.Threading.Tasks.Task<Item.Ilmu.ObjectService.LogErrorCodes> MovementMethod1_ManualWithTorqueLimit(double Position, double Speed, double Acceleration, double TorqueLimit)
        {
            // Specs for ComObject 0x05A3 (ControlWord2)
            // Criteria for target reached:
            // 0x0   = Standard (Position reached)
            // Bit 8   0x100 = Target reached if Torque Limit (Current) reached  <----------------------- Thats what we want
            // Bit 9   0x200 = Target reached if n_ist = 0
            // Bit 10  0x400 = Target reached if Torque Limit (Regulator Torque) reached

            // First we read the current ControlWord, because we only want to change one parameter
            Item.Ilmu.ObjectService.PositionControlword2 ControlWord2 = (Item.Ilmu.ObjectService.PositionControlword2)await Controller.GetComObjectValueAsync<double>(Item.Ilmu.ObjectService.ComObjectType.POSITIONPOINTER, 0, Item.Ilmu.ObjectService.ComObjectType.POSITIONCONTROLWORD2);

            ControlWord2 |= Item.Ilmu.ObjectService.PositionControlword2.TargetReachedCriteria1; // Set Bit8
            ControlWord2 &= ~(Item.Ilmu.ObjectService.PositionControlword2.TargetReachedCriteria2 | Item.Ilmu.ObjectService.PositionControlword2.TargetReachedCriteria3); // Unset Bit9 & 10

            // This will set the ControlWord to 0x28, which means we have another WP after this WP, which is stored in "Successor"
            Item.Ilmu.ObjectService.PositionControlword ControlWord = (Item.Ilmu.ObjectService.PositionControlword.StartWPPositioningMode1 | Item.Ilmu.ObjectService.PositionControlword.StartWPPositioningMode2 | Item.Ilmu.ObjectService.PositionControlword.StartWPPositioningMode3);

            // Here we manually write the position data into WP 0 because we also want to set up a torque limit, which is
            // not useful if there is no next point in the waypoint list, as the axis will just get stuck at the end
            await Controller.WritePositionDataAsync(0, new Item.Ilmu.ObjectService.PositionInfo()
            {
                Endpos = Position,
                Accel = Acceleration,
                Decel = Acceleration,
                Speed = Speed,
                Controlword2 = ControlWord2,
                TorqueThreshold = TorqueLimit,
                Successor = 1,
                Controlword = ControlWord,
            });

            // If it hits the torque limit or arrives at the position it will move to the next waypoint, which is this one
            await Controller.WritePositionDataAsync(1, new Item.Ilmu.ObjectService.PositionInfo()
            {
                Endpos = 0,
                Accel = Acceleration,
                Decel = Acceleration,
                Speed = Speed * 2,
                Controlword2 = ControlWord2,
                TorqueThreshold = TorqueLimit,
                Successor = 0,
                Controlword = 0,
            });

            // This call starts the waypoint list
            Item.Ilmu.ObjectService.LogErrorCodes f = await Controller.SendMovementCommandToControllerAsync(Item.Ilmu.ObjectService.MovementMode.MoveToWaypoint, 0);

            return f;
        }

        /// <summary>
        /// This will just move to the position using the standard special WP 264. But you cannot set any options like torque limits
        /// </summary>
        /// <param name="Position"></param>
        /// <param name="Speed"></param>
        /// <param name="Acceleration"></param>
        /// <returns></returns>
        private async System.Threading.Tasks.Task<Item.Ilmu.ObjectService.LogErrorCodes> MovementMethod2_AutomaticPositioning(double Position, double Speed, double Acceleration)
        {
            // This call will automatically set the values into position 264 and execute
            return await Controller.SendMovementCommandToControllerAsync(Item.Ilmu.ObjectService.MovementMode.MoveToPosition, Position, Speed, Acceleration);

            // Same as above, but sync. Do not use this in a GUI program, as it will only return after the position was reached, which can take a long time
            //Item.Ilmu.ObjectService.LogErrorCodes f = Controller.SendMovementCommandToController(Item.Ilmu.ObjectService.MovementMode.MoveToPosition, Position, Speed, Acceleration);
        }

        private async void ButtonEnable_Click(object sender, RoutedEventArgs e)
        {
            // You need to clear errors before you can enable the controller
            await Controller.ClearWarningsAndErrorsAsync();

            // Make sure the PC can enable the controller
            await Controller.SetClearanceSelectorAsync(Item.Ilmu.ObjectService.ReglerFreigabeSelector.DIN5AndPC);

            // How to enable the controller
            var error = await Controller.SendDeviceCommandToControllerAsync(Item.Ilmu.ObjectService.IlmuCmd.EnableController);

            if (error == Item.Ilmu.ObjectService.LogErrorCodes.CommandTimeout)
            {
                Debug.WriteLine("Controller can not be enabled. Check if Din4 + Din5 are set up properly.");
            }
        }

        private async void ButtonDisable_Click(object sender, RoutedEventArgs e)
        {
            // How to disable the controller
            await Controller.SendDeviceCommandToControllerAsync(Item.Ilmu.ObjectService.IlmuCmd.DisableController);
        }

        private async void ButtonReference_Click(object sender, RoutedEventArgs e)
        {
            // This will start a referencing process
            await Controller.StartReferencingAsync(Item.Ilmu.ObjectService.ReferenceDirection.CURRENTLY_SET);
        }

        private async void ButtonStop_Click(object sender, RoutedEventArgs e)
        {
            await Controller.SendMovementStopToControllerAsync();
        }

        #endregion

        #region Helper

        private void UpdateConnectionStatus()
        {
            if (!Dispatcher.CheckAccess())
            {
                Dispatcher.Invoke(() => UpdateConnectionStatus());
                return;
            }

            if (Controller != null && Controller.IsConnected)
                ButtonConnect.Background = new SolidColorBrush(Color.FromRgb(0, 255, 0));
            else
                ButtonConnect.Background = new SolidColorBrush(Color.FromRgb(255, 0, 0));
        }

        #endregion
    }
}
